<?php
// updater.php
if (!defined('ABSPATH')) {
    exit;
}

class CBW_Plugin_Updater {
    private $file;
    private $plugin;
    private $basename;
    private $active;
    private $update_server;
    private $cache_key;
    private $cache_expiration = DAY_IN_SECONDS; // 24 hours
    
    public function __construct($file) {
        $this->file = $file;
        $this->update_server = 'https://thepervtube.com/cbwidgetcams/updater.json';
        
        // Set plugin properties
        $this->set_plugin_properties();
        
        // Set cache key
        $this->cache_key = 'cbw_update_info_' . md5($this->basename);
        
        error_log('CBW_Plugin_Updater (self-hosted) initialized for: ' . $this->basename);
    }
    
    public function set_plugin_properties() {
        $this->plugin = get_plugin_data($this->file);
        $this->basename = plugin_basename($this->file);
        $this->active = is_plugin_active($this->basename);
    }
    
    public function initialize() {
        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_update'));
        add_filter('plugins_api', array($this, 'plugins_api_handler'), 10, 3);
        add_filter('upgrader_post_install', array($this, 'after_install'), 10, 3);
        
        error_log('CBW_Plugin_Updater (self-hosted) hooks registered');
    }
    
    private function get_remote_info() {
        // Check cache first
        $cached_info = get_transient($this->cache_key);
        if ($cached_info !== false) {
            error_log('Using cached update info');
            return $cached_info;
        }
        
        error_log('Fetching update info from: ' . $this->update_server);
        
        $response = wp_remote_get($this->update_server, array(
            'timeout' => 15,
            'headers' => array(
                'Accept' => 'application/json'
            )
        ));
        
        if (is_wp_error($response)) {
            error_log('Update server error: ' . $response->get_error_message());
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            error_log('Update server returned HTTP ' . $response_code);
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $update_info = json_decode($body, true);
        
        if (!is_array($update_info) || !isset($update_info['version'])) {
            error_log('Invalid update info format');
            return false;
        }
        
        // Cache the response for 24 hours
        set_transient($this->cache_key, $update_info, $this->cache_expiration);
        
        error_log('Update info fetched and cached: Version ' . $update_info['version']);
        return $update_info;
    }
    
    public function check_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }
        
        $remote_info = $this->get_remote_info();
        
        if ($remote_info && version_compare($remote_info['version'], $this->plugin['Version'], '>')) {
            $package = $remote_info['download_url'];
            
            $update_data = array(
                'slug' => 'chaturbate-affiliate-cams-widget',
                'plugin' => $this->basename,
                'new_version' => $remote_info['version'],
                'url' => $this->plugin['PluginURI'],
                'package' => $package,
                'tested' => $remote_info['tested'] ?? get_bloginfo('version'),
                'requires_php' => $remote_info['requires_php'] ?? '7.4',
                'requires' => $remote_info['requires'] ?? '5.0'
            );
            
            if (!isset($transient->response)) {
                $transient->response = array();
            }
            
            $transient->response[$this->basename] = (object) $update_data;
            error_log('✅ UPDATE AVAILABLE: Version ' . $remote_info['version'] . ' for ' . $this->basename);
        } else {
            error_log('No update needed - Current: ' . $this->plugin['Version'] . ', Remote: ' . ($remote_info['version'] ?? 'unknown'));
        }
        
        return $transient;
    }
    
    public function plugins_api_handler($result, $action, $args) {
        if ($action !== 'plugin_information') {
            return $result;
        }
        
        if (!isset($args->slug) || $args->slug !== 'chaturbate-affiliate-cams-widget') {
            return $result;
        }
        
        $remote_info = $this->get_remote_info();
        
        if ($remote_info) {
            $plugin_info = array(
                'name' => $this->plugin['Name'],
                'slug' => 'chaturbate-affiliate-cams-widget',
                'version' => $remote_info['version'],
                'author' => $this->plugin['AuthorName'],
                'author_profile' => $this->plugin['AuthorURI'],
                'last_updated' => $remote_info['last_updated'] ?? date('Y-m-d H:i:s'),
                'homepage' => $this->plugin['PluginURI'],
                'short_description' => $this->plugin['Description'],
                'sections' => array(
                    'description' => $remote_info['sections']['description'] ?? $this->plugin['Description'],
                    'changelog' => $remote_info['sections']['changelog'] ?? 'No changelog available.',
                    'installation' => $remote_info['sections']['installation'] ?? ''
                ),
                'download_link' => $remote_info['download_url'],
                'banners' => array(
                    'low' => $remote_info['banners']['low'] ?? '',
                    'high' => $remote_info['banners']['high'] ?? ''
                )
            );
            
            return (object) $plugin_info;
        }
        
        return $result;
    }
    
    public function after_install($response, $hook_extra, $result) {
        // Only process if this is our plugin
        if (!isset($hook_extra['plugin']) || $hook_extra['plugin'] !== $this->basename) {
            return $response;
        }
        
        error_log('After install completed for: ' . $hook_extra['plugin']);
        
        // Clear the cache to ensure WordPress recognizes the new version
        wp_clean_plugins_cache();
        
        // Clear our update cache
        delete_transient($this->cache_key);
        
        // Reactivate the plugin if it was active
        if ($this->active) {
            error_log('Reactivating plugin: ' . $this->basename);
            activate_plugin($this->basename);
        }
        
        return $result;
    }
}